/*
 * Copyright 2019 NXP
 * All rights reserved.
 * SPDX-License-Identifier: BSD-3-Clause
 *
 * This module manages the reception TSCP communications using a UART interface
 */

/*==================================================================================================
 Include Files
==================================================================================================*/
#include "EmbeddedTypes.h"
#include "tscp.h"
#include "TimersManager.h"
#include "serial_manager.h"
#include "FunctionLib.h"

/*==================================================================================================
 Private macros
==================================================================================================*/
#ifndef TSCP_PC_BUFFER_SIZE
#define TSCP_PC_BUFFER_SIZE (128U)
#endif

/*==================================================================================================
 Private global variables declarations
==================================================================================================*/
static uint8_t gTscpAppSerId;
/* Indexed by TSCP instanceId. Allow convertion from AppId(in network packet) to instanceId(internal TSCP index) */
static uint8_t gTscpAppIds[TSCP_MAX_NUMBER_OF_INSTANCES];
static tmrTimeInSeconds_t gTscpMaxWaitTimeInSec;

/* UART interface timer */
static tmrTimerID_t gTscpUartTimeoutTimerId;

/*==================================================================================================
 Private functions
==================================================================================================*/
static void TSCP_UartTscpCommunicationsHandler (bool_t forceEngineReset);

static void TSCP_UartTscpTimeoutCallback (void* pParam)
{
    TSCP_UartTscpCommunicationsHandler(TRUE);
}

static void TSCP_SerialCallback(void* param)
{
    TSCP_UartTscpCommunicationsHandler(FALSE);
}

static void TSCP_UartTscpCommunicationsHandler (bool_t forceEngineReset){
    static uint8_t uartReceiveBuffer[TSCP_PC_BUFFER_SIZE];
    static uint16_t payloadToWait = 0u;
    tscpPacketStructureHeader_t* pTscpHeader;
    uint16_t receivedBytes, bytesRead;
    bool_t engineReset = FALSE;

    /* Check how many bytes have been received */
    (void)Serial_RxBufferByteCount(gTscpAppSerId, &receivedBytes);

    if(payloadToWait == 0u)
    {
        if(receivedBytes >= TSCP_HEADER_OFFSET)
        {
            /* At least 6 bytes have been received (Header + Opcode + Payload). Read them */
            (void)Serial_Read(gTscpAppSerId, uartReceiveBuffer, TSCP_HEADER_OFFSET, &bytesRead);

            if(bytesRead == TSCP_HEADER_OFFSET)
            {
                pTscpHeader = (tscpPacketStructureHeader_t*)(void *)uartReceiveBuffer;
                if (pTscpHeader->startHeader == TSCP_START_CODE_PARAMS)
                {
                    /* This is a params header. Read the Payload and wait for it*/
                    payloadToWait = 2U * pTscpHeader->lenght;
                }
                if (pTscpHeader->startHeader == TSCP_START_CODE_RAW)
                {
                    /* This is a raw header. Read the Payload and wait for it*/
                    payloadToWait = pTscpHeader->lenght;
                }
            }
        }
    }
    else
    {
        uint8_t instance = 0xFFu;

        if(receivedBytes >= payloadToWait)
        {
            /* Stop the Timeout Timer */
            (void)TMR_StopTimer(gTscpUartTimeoutTimerId);

            /* Read the payload */
            (void)Serial_Read(gTscpAppSerId, &uartReceiveBuffer[TSCP_HEADER_OFFSET], payloadToWait, &bytesRead);

            pTscpHeader = (tscpPacketStructureHeader_t*)(void *)uartReceiveBuffer;

            /* Retreive instance from AppId in header */
            for (uint32_t i = 0; i < TSCP_MAX_NUMBER_OF_INSTANCES; i++)
            {
                if (gTscpAppIds[i] == pTscpHeader->appId)
                {
                    instance = (uint8_t)i;
                }
            }

            if (instance != 0xFFu)
            {
                /* Parse Packet, invoke callback */
                TSCP_Parse(instance, uartReceiveBuffer);
            }
            else
            {
                /* Unknown applicationId */
            }

            /* Assert engine reset */
            engineReset = TRUE;
        }
    }

    if (engineReset || forceEngineReset)
    {
        /* Reset the payload to wait */
        payloadToWait = 0u;

        /* Flush RX Buffer */
        (void)Serial_RxBufferByteCount(gTscpAppSerId, &receivedBytes);

        if (receivedBytes != 0u)
        {
            (void)Serial_Read(gTscpAppSerId, uartReceiveBuffer, receivedBytes, &bytesRead);
        }

        /* Reset Buffer */
        FLib_MemSet(uartReceiveBuffer, 0x00u, TSCP_PC_BUFFER_SIZE);
    }
    else
    {
        /*Restart Timeout Timer */
        (void)TMR_StopTimer(gTscpUartTimeoutTimerId);
        (void)TMR_StartLowPowerSecondTimer(gTscpUartTimeoutTimerId, gTscpMaxWaitTimeInSec, TSCP_UartTscpTimeoutCallback, NULL);
    }
}

/*==================================================================================================
 Public functions
==================================================================================================*/
void TSCP_UartInit(uint8_t appSerId, tmrTimeInSeconds_t maxWaitTimeInSec)
{
    gTscpAppSerId = appSerId;
    gTscpMaxWaitTimeInSec = maxWaitTimeInSec;

    /* Configure Serial Manager Timeout Timer */
    gTscpUartTimeoutTimerId = TMR_AllocateTimer();
    assert(gTscpUartTimeoutTimerId != gTmrInvalidTimerID_c); //Error: Timer couldn't be allocated

    /*set Serial Manager receive callback*/
    (void)Serial_SetRxCallBack(appSerId, TSCP_SerialCallback, NULL);
}

serialStatus_t TSCP_UartBind(uint8_t appId, uint8_t instance)
{
    serialStatus_t status = gSerial_Success_c;
    if (instance < TSCP_MAX_NUMBER_OF_INSTANCES)
    {
        /* Link an appId (appearing in packet header) to a TSCP instance (internal representation) */
        gTscpAppIds[instance] = appId;
    }
    else
    {
        status = gSerial_InvalidParameter_c;
    }
    return status;
}

serialStatus_t TSCP_UartSyncWriteParams(uint8_t appId, uint8_t opCode, int16_t *pParamList, uint16_t paramCount)
{
    serialStatus_t status;
    uint8_t* pMessageArray;
    uint16_t messageSize;

    pMessageArray = TSCP_PackParams(appId, opCode, pParamList, paramCount, &messageSize);
    if(pMessageArray != NULL)
    {
        status = Serial_SyncWrite(gTscpAppSerId, pMessageArray, messageSize);
        TSCP_PackDestroy(pMessageArray);
    }
    else
    {
        status = gSerial_OutOfMemory_c;
    }
    return status;
}

serialStatus_t TSCP_UartSyncWriteParamsList(uint8_t instance, uint8_t opCode, uint8_t nbListEntries, tscpParamList_t *pParamListList)
{
    serialStatus_t status;
    uint8_t* pMessageArray;
    uint16_t messageSize;

    pMessageArray = TSCP_PackParamsList(instance, opCode, nbListEntries, pParamListList, &messageSize);
    
    if (pMessageArray != NULL)
    {
        status = Serial_SyncWrite(gTscpAppSerId, pMessageArray, messageSize);
        TSCP_PackDestroy(pMessageArray);
    }
    else
    {
        status = gSerial_OutOfMemory_c;
    }
    return status;
}

serialStatus_t TSCP_UartSyncWriteRaw(uint8_t appId, uint8_t opCode, uint8_t *pbuffer, uint16_t bufLen)
{
    serialStatus_t status;
    tscpPacketStructureHeader_t header;

    /* Write TSCP header */
    header.startHeader = TSCP_START_CODE_RAW;
    header.appId = appId;
    header.opCode = opCode;
    header.lenght = bufLen;
    status = Serial_SyncWrite(gTscpAppSerId, (uint8_t *)&header, (uint16_t)sizeof(header));

    /* Write payload directly to serail link */
    if (status == gSerial_Success_c)
    {
        status = Serial_SyncWrite(gTscpAppSerId, pbuffer, bufLen);
    }

    return status;
}
